C      *****************************************************************
C      * Subroutine ScaleOrder                                         *
C      * Finds the largest number in each row by searching each column *
C      * in the row for the largest absolute value and then divides    *
C      * the entire row by this value.  This helps reduce roundoff     *
C      * errors in subsequent steps.  This routine also initializes    *
C      * the ORDER array to indicate that the largest coefficients     *
C      * for each column are in the row of the same number.  This is a *
C      * likely order for the matrix and will therefore speed up       *
C      * calculations when it is correct.                              *
C      *                                                               *
C      * Variables:                                                    *
C      * A         -Input - A matrix of size "Size" by ("Size"+1).     *
C      *                    The "Size" by "Size" portion of the matrix *
C      *                    contains the coefficents of "Size"         *
C      *                    equations in "Size" unknowns, and the      *
C      *                    "Size"+1st column is the right hand side   *
C      *                    vector of constant terms.  "Size" is equal *
C      *                    to the number of components in the system. *
C      *                    Units are consistent with inputs.          *
C      *                    (Common block VGauss, file VGauss.f).      *
C      * Col       -Local - Column index into matrix A.  Unitless.     *
C      * Order     -Output- A vector of size "Size" which orders the   *
C      *                    rows of A so that in this order A is an    *
C      *                    upper triangular matrix.  Unitless.        *
C      *                    (Common block VGauss, file VGauss.f).      *
C      * Row       -Local - Row index into matrix A.  Unitless.        *
C      * Scale     -Local - The largest value in the current row.      *
C      *                    Units consistent with matrix A.            *
C      * Size      -Input - The number of chemical components in the   *
C      *                    system.  Unitless.                         *
C      *                    (Common block VGauss, file VGauss.f).      *
C      *****************************************************************
 	     SUBROUTINE SCALEORDER
							IMPLICIT NONE
							INCLUDE 'VGAUSS.INC'
   	   INTEGER COL, ROW
							REAL*8 SCALE
	  
C      *-------------------------------------------------------------*
C      * Initialize Order array and initalize the scale value to the *
C      * first item in the row.                                      *
C      *-------------------------------------------------------------*
   	   DO 100 ROW=1, SIZE
   	      ORDER(ROW) = ROW
   	   	  SCALE = ABS(A(ROW,1))
		  
C         *------------------------------------*
C         * Find the largest value in this row *
C         *------------------------------------*
		        DO 110 COL=2, SIZE+1
		           IF (ABS(A(ROW,COL)).GT.SCALE) THEN
			             SCALE = ABS(A(ROW,COL))
		         	 ENDIF
  110     CONTINUE

C         *-------------------------------------------------------*
C         * Scale all values in the row to this value so that the *
C         * largest value is "1".                                 *
C         *-------------------------------------------------------*
          DO 120 COL=1,SIZE+1
		           A(ROW,COL) = A(ROW,COL)/SCALE
  120     CONTINUE
  
  100  CONTINUE

       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
